<?php
/**
 * LayerPay Convert Controller
 * Handles price conversion from fiat to crypto
 */

class LayerPayConvertModuleFrontController extends ModuleFrontController
{
    public function initContent()
    {
        parent::initContent();

        header('Content-Type: application/json');

        $amount = (float)Tools::getValue('amount', 0);
        $currency = strtolower(Tools::getValue('currency', 'eur'));
        $type = Tools::getValue('type', 'eth'); // eth or usdc

        if ($amount <= 0) {
            $this->jsonResponse(['success' => false, 'error' => 'Invalid amount']);
        }

        $margin = (float)Configuration::get('LAYERPAY_PRICE_MARGIN');
        $cacheTime = (int)Configuration::get('LAYERPAY_PRICE_CACHE');

        try {
            if ($type === 'usdc') {
                $result = $this->convertToUsdc($amount, $currency, $margin);
            } else {
                $result = $this->convertToEth($amount, $currency, $margin, $cacheTime);
            }

            $this->jsonResponse([
                'success' => true,
                'data' => $result
            ]);
        } catch (Exception $e) {
            $this->jsonResponse([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    protected function convertToEth($amount, $currency, $margin, $cacheTime)
    {
        // Get ETH price
        $ethPrice = $this->getEthPrice($currency, $cacheTime);

        if (!$ethPrice) {
            throw new Exception('Failed to fetch ETH price');
        }

        // Apply margin
        $effectivePrice = $ethPrice * (1 - $margin / 100);

        // Calculate ETH amount
        $ethAmount = $amount / $effectivePrice;

        // Convert to wei (18 decimals)
        $weiAmount = bcmul(number_format($ethAmount, 18, '.', ''), '1000000000000000000', 0);

        return [
            'eth_amount' => number_format($ethAmount, 8, '.', ''),
            'wei_amount' => $weiAmount,
            'eth_price' => number_format($ethPrice, 2, '.', ''),
            'effective_price' => number_format($effectivePrice, 2, '.', ''),
            'margin_percent' => $margin,
            'currency' => strtoupper($currency)
        ];
    }

    protected function convertToUsdc($amount, $currency, $margin)
    {
        // USDC is pegged 1:1 to USD
        $usdAmount = $amount;

        // Convert from EUR if needed
        if ($currency === 'eur') {
            $rate = $this->getExchangeRate('EUR', 'USD');
            if (!$rate) {
                throw new Exception('Failed to fetch EUR/USD exchange rate');
            }
            $usdAmount = $amount * $rate;
        }

        // Apply margin
        $usdcAmount = $usdAmount * (1 + $margin / 100);

        // USDC has 6 decimals
        $usdcSmallestUnit = bcmul(number_format($usdcAmount, 6, '.', ''), '1000000', 0);

        return [
            'usdc_amount' => number_format($usdcAmount, 2, '.', ''),
            'usdc_smallest_unit' => $usdcSmallestUnit,
            'usd_amount' => number_format($usdAmount, 2, '.', ''),
            'margin_percent' => $margin,
            'currency' => strtoupper($currency)
        ];
    }

    protected function getEthPrice($currency, $cacheTime)
    {
        $cacheKey = 'layerpay_eth_price_' . $currency;

        // Check cache
        $cached = Cache::getInstance()->get($cacheKey);
        if ($cached !== false) {
            return (float)$cached;
        }

        // Fetch from CoinGecko
        $price = $this->fetchFromCoinGecko($currency);

        // Fallback to CryptoCompare
        if (!$price) {
            $price = $this->fetchFromCryptoCompare($currency);
        }

        if ($price) {
            Cache::getInstance()->set($cacheKey, $price, $cacheTime);
        }

        return $price;
    }

    protected function fetchFromCoinGecko($currency)
    {
        $url = sprintf(
            'https://api.coingecko.com/api/v3/simple/price?ids=ethereum&vs_currencies=%s',
            strtolower($currency)
        );

        $response = $this->httpGet($url);
        if (!$response) {
            return null;
        }

        $data = json_decode($response, true);
        return $data['ethereum'][$currency] ?? null;
    }

    protected function fetchFromCryptoCompare($currency)
    {
        $url = sprintf(
            'https://min-api.cryptocompare.com/data/price?fsym=ETH&tsyms=%s',
            strtoupper($currency)
        );

        $response = $this->httpGet($url);
        if (!$response) {
            return null;
        }

        $data = json_decode($response, true);
        return $data[strtoupper($currency)] ?? null;
    }

    protected function getExchangeRate($from, $to)
    {
        $cacheKey = 'layerpay_rate_' . $from . '_' . $to;

        $cached = Cache::getInstance()->get($cacheKey);
        if ($cached !== false) {
            return (float)$cached;
        }

        $url = sprintf('https://api.exchangerate-api.com/v4/latest/%s', $from);
        $response = $this->httpGet($url);

        if (!$response) {
            return null;
        }

        $data = json_decode($response, true);
        $rate = $data['rates'][$to] ?? null;

        if ($rate) {
            Cache::getInstance()->set($cacheKey, $rate, 3600); // Cache for 1 hour
        }

        return $rate;
    }

    protected function httpGet($url)
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'User-Agent: LayerPay/1.0'
            ]
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($error || $httpCode !== 200) {
            return null;
        }

        return $response;
    }

    protected function jsonResponse($data)
    {
        die(json_encode($data));
    }
}
