/**
 * Layer Crypto Checkout - Wallet Integration (MetaMask + WalletConnect)
 *
 * Handles wallet connection, price conversion, and payment processing
 * Supports ETH and USDC payments on multiple blockchain networks
 * Uses Web3Modal v1.x for multi-wallet support
 */

(function($) {
    'use strict';

    // State management
    const LCC = {
        isConnected: false,
        account: null,
        ethAmount: null,
        weiAmount: null,
        usdcAmount: null,
        usdcSmallestUnit: null,
        orderId: null,
        initialized: false,
        selectedNetwork: null,
        paymentType: 'eth', // 'eth' or 'usdc'
        connectionType: null, // 'metamask' or 'walletconnect'
        walletConfig: null,
        wcProviderClass: null,
        wcProvider: null,
        provider: null,
        listenersSetup: false,

        /**
         * Initialize the payment handler
         */
        init: function() {
            const self = this;

            // Initialize Web3Modal for WalletConnect support
            this.initWeb3Modal();

            // Use event delegation for dynamic elements
            $(document).on('click', '#lcc-connect-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.handleConnectClick();
            });

            // Disconnect button
            $(document).on('click', '#lcc-disconnect-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.disconnect();
            });

            // Listen for network selector change
            $(document).on('change', '#lcc-network', function() {
                const networkKey = $(this).val();
                self.onNetworkChange(networkKey);
            });

            // Listen for payment type change
            $(document).on('change', 'input[name="lcc_payment_type"]', function() {
                const paymentType = $(this).val();
                self.onPaymentTypeChange(paymentType);
            });

            // Listen for payment method change
            $(document).on('change', 'input[name="payment_method"]', function() {
                if ($(this).val() === 'layer-crypto-checkout') {
                    self.onPaymentMethodSelected();
                }
            });

            // Check if already selected on page load
            $(document).ready(function() {
                setTimeout(function() {
                    if ($('input[name="payment_method"]:checked').val() === 'layer-crypto-checkout') {
                        self.onPaymentMethodSelected();
                    }
                }, 500);
            });

            // Listen for checkout updates
            $(document.body).on('updated_checkout', function() {
                if ($('input[name="payment_method"]:checked').val() === 'layer-crypto-checkout') {
                    self.onPaymentMethodSelected();
                }
            });

            // Form validation
            $(document).on('checkout_place_order_layer-crypto-checkout', function() {
                return self.validatePayment();
            });

        },

        /**
         * Initialize wallet connection options
         */
        initWeb3Modal: function() {
            const self = this;

            // Check if WalletConnect libraries are loaded
            const checkLibraries = function() {
                if (window.LCCWalletConfig) {
                    self.walletConfig = window.LCCWalletConfig;
                    console.log('LCC: Wallet config ready');

                    // Check for WalletConnect ethereum provider
                    if (window.EthereumProvider) {
                        self.wcProviderClass = window.EthereumProvider.EthereumProvider || window.EthereumProvider;
                        console.log('LCC: WalletConnect provider available');
                    }

                    return true;
                }
                return false;
            };

            // Try immediately
            if (!checkLibraries()) {
                let attempts = 0;
                const interval = setInterval(function() {
                    attempts++;
                    if (checkLibraries() || attempts >= 10) {
                        clearInterval(interval);
                    }
                }, 500);
            }

            // Create wallet selection modal
            this.createWalletModal();
        },

        /**
         * Create the wallet selection modal HTML
         */
        createWalletModal: function() {
            if ($('#lcc-wallet-modal').length) return;

            const modalHtml = `
                <div id="lcc-wallet-modal" style="display:none; position:fixed; top:0; left:0; right:0; bottom:0; background:rgba(0,0,0,0.5); z-index:99999; align-items:center; justify-content:center;">
                    <div style="background:white; border-radius:16px; padding:24px; max-width:360px; width:90%; box-shadow:0 10px 40px rgba(0,0,0,0.2);">
                        <h3 style="margin:0 0 20px; font-size:18px; text-align:center;">Connect Wallet</h3>
                        <button id="lcc-connect-metamask" style="width:100%; padding:14px; margin-bottom:12px; border:1px solid #e5e5e5; border-radius:12px; background:white; cursor:pointer; display:flex; align-items:center; gap:12px; font-size:16px; transition:background 0.2s;">
                            <img src="${lccData.pluginUrl}assets/images/metamask.svg" style="width:32px; height:32px;">
                            <span>MetaMask</span>
                        </button>
                        <button id="lcc-connect-walletconnect" style="width:100%; padding:14px; border:1px solid #e5e5e5; border-radius:12px; background:white; cursor:pointer; display:flex; align-items:center; gap:12px; font-size:16px; transition:background 0.2s;">
                            <img src="${lccData.pluginUrl}assets/images/walletconnect.png" style="width:32px; height:32px; border-radius:8px;">
                            <span>WalletConnect</span>
                        </button>
                        <button id="lcc-modal-close" style="width:100%; padding:10px; margin-top:16px; border:none; background:none; cursor:pointer; color:#666; font-size:14px;">Cancel</button>
                    </div>
                </div>
            `;
            $('body').append(modalHtml);

            // Add hover effects
            $('#lcc-connect-metamask, #lcc-connect-walletconnect').hover(
                function() { $(this).css('background', '#f5f5f5'); },
                function() { $(this).css('background', 'white'); }
            );

            // Event handlers
            const self = this;
            $('#lcc-wallet-modal').on('click', function(e) {
                if (e.target === this) self.hideWalletModal();
            });
            $('#lcc-modal-close').on('click', function() {
                self.hideWalletModal();
            });
            $('#lcc-connect-metamask').on('click', function() {
                self.hideWalletModal();
                self.connectMetaMask();
            });
            $('#lcc-connect-walletconnect').on('click', function() {
                self.hideWalletModal();
                self.connectWalletConnect();
            });
        },

        /**
         * Show wallet selection modal
         */
        showWalletModal: function() {
            $('#lcc-wallet-modal').css('display', 'flex');
        },

        /**
         * Hide wallet selection modal
         */
        hideWalletModal: function() {
            $('#lcc-wallet-modal').hide();
            this.setButtonState('connect');
        },

        /**
         * Get the currently selected network configuration
         */
        getSelectedNetwork: function() {
            const networkKey = $('#lcc-network').val() || lccData.defaultNetwork || 'sepolia';
            const network = lccData.networks[networkKey];
            if (!network) {
                console.error('LCC: Network not found:', networkKey);
                return lccData.networks['sepolia'] || {};
            }
            return {
                key: networkKey,
                name: network.name,
                chainId: network.chainId,
                chainIdDec: network.chainIdDec,
                contract: network.contract,
                rpcUrl: network.rpcUrl,
                explorer: network.explorer,
                symbol: network.symbol,
                isTestnet: network.isTestnet,
                usdcAddress: network.usdcAddress,
                usdcDecimals: network.usdcDecimals || 6
            };
        },

        /**
         * Called when network selector changes
         */
        onNetworkChange: async function(networkKey) {
            const self = this;
            const network = lccData.networks[networkKey];
            if (!network) {
                console.error('LCC: Network not found:', networkKey);
                return;
            }

            this.selectedNetwork = network;

            // Reset payment state
            this.ethAmount = null;
            this.weiAmount = null;
            this.usdcAmount = null;
            this.usdcSmallestUnit = null;

            // If connected, switch network and refresh conversion
            if (this.isConnected && this.provider) {
                try {
                    await self.ensureCorrectNetwork();
                    await self.fetchConversion();
                } catch (err) {
                    console.error('LCC: Network switch failed:', err);
                    self.showError('Failed to switch network. Please switch manually in your wallet.');
                }
            }
        },

        /**
         * Called when payment type changes (ETH vs USDC)
         */
        onPaymentTypeChange: async function(paymentType) {
            this.paymentType = paymentType;

            // Update the hidden field for form submission
            $('#lcc-payment-type').val(paymentType);

            // Reset amounts
            this.ethAmount = null;
            this.weiAmount = null;
            this.usdcAmount = null;
            this.usdcSmallestUnit = null;

            // Update button text
            this.updateButtonText();

            // Refresh conversion if connected
            if (this.isConnected) {
                await this.fetchConversion();
            }
        },

        /**
         * Sync network selector when user changes chain in wallet
         */
        syncNetworkSelector: function(chainId) {
            const networks = lccData.networks;
            for (const key in networks) {
                if (networks[key].chainId === chainId) {
                    $('#lcc-network').val(key);
                    this.selectedNetwork = networks[key];
                    this.fetchConversion();
                    return;
                }
            }
            this.showError('Please select a supported network.');
        },

        /**
         * Called when Layer Crypto Checkout payment method is selected
         */
        onPaymentMethodSelected: function() {
            // Initialize selected network
            const network = this.getSelectedNetwork();
            this.selectedNetwork = network;

            // Check current payment type
            const paymentType = $('input[name="lcc_payment_type"]:checked').val() || 'eth';
            this.paymentType = paymentType;

            // Check if already connected (e.g., from previous session)
            this.checkExistingConnection();

            if (this.isConnected) {
                this.updateUI();
                this.fetchConversion();
            }
        },

        /**
         * Check for existing wallet connection
         */
        checkExistingConnection: function() {
            const self = this;

            // Web3Modal handles cached provider reconnection in initWeb3Modal
            // Here we just check for injected provider
            if (window.ethereum) {
                let provider = null;

                if (window.ethereum.providers && window.ethereum.providers.length) {
                    provider = window.ethereum.providers.find(p => p.isMetaMask);
                } else if (window.ethereum.isMetaMask) {
                    provider = window.ethereum;
                } else {
                    provider = window.ethereum;
                }

                if (provider) {
                    this.provider = provider;
                    this.setupProviderListeners();

                    // Check if already connected
                    provider.request({ method: 'eth_accounts' }).then(function(accounts) {
                        if (accounts && accounts.length > 0) {
                            self.account = accounts[0];
                            self.isConnected = true;
                            self.updateUI();
                            self.fetchConversion();
                        }
                    }).catch(function() {
                        // Not connected, that's fine
                    });
                }
            }
        },

        /**
         * Setup provider event listeners
         */
        setupProviderListeners: function() {
            const self = this;

            if (this.provider && !this.listenersSetup) {
                if (this.provider.on) {
                    this.provider.on('accountsChanged', function(accounts) {
                        if (accounts.length === 0) {
                            self.disconnect();
                        } else {
                            self.account = accounts[0];
                            self.updateUI();
                        }
                    });

                    this.provider.on('chainChanged', function(chainId) {
                        self.syncNetworkSelector(chainId);
                    });

                    this.provider.on('disconnect', function() {
                        self.disconnect();
                    });
                }

                this.listenersSetup = true;
            }
        },

        /**
         * Get DOM elements (fresh lookup each time)
         */
        getElements: function() {
            return {
                container: $('#lcc-payment-container'),
                connectBtn: $('#lcc-connect-btn'),
                networkSelect: $('#lcc-network'),
                walletStatus: $('#lcc-wallet-status'),
                priceDisplay: $('#lcc-price-display'),
                cryptoAmountEl: $('#lcc-crypto-amount'),
                rateEl: $('#lcc-rate'),
                errorEl: $('#lcc-error'),
                txHashInput: $('#lcc-tx-hash'),
                cryptoPaidInput: $('#lcc-eth-paid'),
                walletInput: $('#lcc-wallet-address'),
            };
        },

        /**
         * Handle connect button click
         */
        handleConnectClick: async function() {
            if (!this.isConnected) {
                await this.connect();
            } else if ((this.paymentType === 'eth' && !this.ethAmount) ||
                       (this.paymentType === 'usdc' && !this.usdcAmount)) {
                await this.fetchConversion();
            } else {
                await this.processPayment();
            }
        },

        /**
         * Connect wallet - shows wallet selection modal
         */
        connect: async function() {
            this.hideError();

            // Show wallet selection modal
            this.showWalletModal();
        },

        /**
         * Connect via MetaMask (injected provider)
         */
        connectMetaMask: async function() {
            const self = this;

            try {
                this.setButtonState('connecting');

                if (!window.ethereum) {
                    this.showError(lccData.i18n.noWalletFound);
                    this.setButtonState('connect');
                    return;
                }

                let provider = null;
                if (window.ethereum.providers && window.ethereum.providers.length) {
                    provider = window.ethereum.providers.find(p => p.isMetaMask);
                }
                if (!provider) {
                    provider = window.ethereum;
                }

                this.provider = provider;
                this.setupProviderListeners();

                const accounts = await provider.request({
                    method: 'eth_requestAccounts'
                });

                if (accounts.length === 0) {
                    throw new Error('No accounts found');
                }

                this.account = accounts[0];
                this.isConnected = true;
                this.connectionType = 'metamask';

                await this.onConnectionSuccess();

            } catch (error) {
                console.error('LCC: MetaMask connection error:', error);
                if (error.code === 4001) {
                    // User rejected
                    this.setButtonState('connect');
                } else {
                    this.showError(lccData.i18n.transactionFailed);
                    this.setButtonState('connect');
                }
            }
        },

        /**
         * Connect via WalletConnect
         */
        connectWalletConnect: async function() {
            const self = this;

            try {
                this.setButtonState('connecting');

                // Try different possible export names
                if (!this.wcProviderClass) {
                    var wcModule = window['@walletconnect/ethereum-provider'];
                    if (wcModule) {
                        this.wcProviderClass = wcModule.EthereumProvider || wcModule.default || wcModule;
                    }
                }

                if (!this.wcProviderClass) {
                    this.showError('WalletConnect not available. Please use MetaMask.');
                    this.setButtonState('connect');
                    return;
                }

                const config = this.walletConfig || window.LCCWalletConfig;
                if (!config || !config.projectId) {
                    this.showError('WalletConnect configuration missing.');
                    this.setButtonState('connect');
                    return;
                }

                // Initialize WalletConnect provider
                const provider = await this.wcProviderClass.init({
                    projectId: config.projectId,
                    chains: [config.defaultChainId],
                    optionalChains: config.chainIds,
                    showQrModal: true,
                    metadata: {
                        name: 'Layer Crypto Checkout',
                        description: 'Crypto Payments',
                        url: window.location.origin,
                        icons: []
                    }
                });

                // Enable session (shows QR modal)
                await provider.enable();

                this.provider = provider;
                this.wcProvider = provider;
                this.setupProviderListeners();

                const accounts = await provider.request({ method: 'eth_accounts' });

                if (!accounts || accounts.length === 0) {
                    throw new Error('No accounts found');
                }

                this.account = accounts[0];
                this.isConnected = true;
                this.connectionType = 'walletconnect';

                await this.onConnectionSuccess();

            } catch (error) {
                console.error('LCC: WalletConnect error:', error);
                if (error.message && error.message.includes('User rejected')) {
                    this.setButtonState('connect');
                } else {
                    this.showError('WalletConnect connection failed. Try MetaMask instead.');
                    this.setButtonState('connect');
                }
            }
        },

        /**
         * Legacy connect for direct provider (kept for compatibility)
         */
        connectDirect: async function() {
            const self = this;
            const elements = this.getElements();

            try {
                this.setButtonState('connecting');
                this.hideError();

                if (!window.ethereum) {
                    this.showError(lccData.i18n.noWalletFound);
                    this.setButtonState('connect');
                    return;
                }

                let provider = null;
                if (window.ethereum.providers && window.ethereum.providers.length) {
                    provider = window.ethereum.providers.find(p => p.isMetaMask);
                }
                if (!provider) {
                    provider = window.ethereum;
                }

                this.provider = provider;
                this.setupProviderListeners();

                const accounts = await provider.request({
                    method: 'eth_requestAccounts'
                });

                if (accounts.length === 0) {
                    throw new Error('No accounts found');
                }

                this.account = accounts[0];
                this.isConnected = true;

                await this.onConnectionSuccess();

            } catch (error) {
                console.error('LCC: Connection error:', error);
                if (error.code === 4001) {
                    // User rejected the request
                    this.showError('Connection cancelled by user.');
                } else {
                    this.showError(error.message || 'Connection failed');
                }
                this.setButtonState('connect');
            }
        },

        /**
         * Handle successful connection
         */
        onConnectionSuccess: async function() {
            const elements = this.getElements();

            await this.ensureCorrectNetwork();
            this.updateUI();
            elements.walletInput.val(this.account);
            await this.fetchConversion();
            this.setButtonState('ready');
        },

        /**
         * Ensure we're on the correct network
         */
        ensureCorrectNetwork: async function() {
            if (!this.provider) return;

            const provider = this.provider;
            const network = this.getSelectedNetwork();
            const targetChainId = network.chainId;

            try {
                const currentChainId = await provider.request({ method: 'eth_chainId' });

                if (currentChainId !== targetChainId) {
                    try {
                        await provider.request({
                            method: 'wallet_switchEthereumChain',
                            params: [{ chainId: targetChainId }],
                        });
                    } catch (switchError) {
                        if (switchError.code === 4902) {
                            await provider.request({
                                method: 'wallet_addEthereumChain',
                                params: [{
                                    chainId: targetChainId,
                                    chainName: network.name,
                                    rpcUrls: [network.rpcUrl],
                                    blockExplorerUrls: [network.explorer],
                                    nativeCurrency: {
                                        name: network.symbol,
                                        symbol: network.symbol,
                                        decimals: 18
                                    }
                                }],
                            });
                        } else {
                            throw switchError;
                        }
                    }
                }
            } catch (error) {
                console.warn('LCC: Could not switch network:', error);
            }
        },

        /**
         * Update UI elements
         */
        updateUI: function() {
            const elements = this.getElements();
            const network = this.getSelectedNetwork();

            if (this.isConnected && this.account) {
                const shortAddress = this.account.slice(0, 6) + '...' + this.account.slice(-4);
                elements.walletStatus
                    .removeClass('disconnected')
                    .addClass('connected')
                    .html('<span class="lcc-status-icon">&#9989;</span>' +
                          '<span class="lcc-status-text">Connected: ' + shortAddress + ' (' + network.name + ')</span>');
            } else {
                elements.walletStatus
                    .removeClass('connected')
                    .addClass('disconnected')
                    .html('<span class="lcc-status-icon">&#128274;</span>' +
                          '<span class="lcc-status-text">' + lccData.i18n.connectWallet + '</span>');
            }
        },

        /**
         * Update button text based on payment type
         */
        updateButtonText: function() {
            if (!this.isConnected) return;

            const btn = this.getElements().connectBtn;
            if (this.paymentType === 'usdc') {
                btn.html('<span class="lcc-btn-icon">&#128176;</span> ' + lccData.i18n.payWithUsdc);
            } else {
                const network = this.getSelectedNetwork();
                btn.html('<span class="lcc-btn-icon">&#128176;</span> Pay with ' + (network.symbol || 'ETH'));
            }
        },

        /**
         * Fetch fiat to crypto conversion
         */
        fetchConversion: async function() {
            const elements = this.getElements();
            const network = this.getSelectedNetwork();

            try {
                elements.cryptoAmountEl.text('Loading...');
                elements.rateEl.text('Loading...');

                const cartTotal = this.getCartTotal();

                if (!cartTotal || cartTotal <= 0) {
                    throw new Error('Could not determine cart total. Please refresh the page.');
                }

                let endpoint, symbol;
                if (this.paymentType === 'usdc') {
                    endpoint = 'convert-usdc';
                    symbol = 'USDC';
                } else {
                    endpoint = 'convert';
                    symbol = network.symbol || 'ETH';
                }

                const response = await fetch(
                    lccData.restUrl + endpoint + '?amount=' + cartTotal + '&currency=' + lccData.currency
                );

                const data = await response.json();

                if (!data.success) {
                    throw new Error(data.error || lccData.i18n.conversionError);
                }

                if (this.paymentType === 'usdc') {
                    this.usdcAmount = data.usdc_amount;
                    this.usdcSmallestUnit = data.usdc_smallest_unit;

                    // Validate USDC conversion result
                    if (!this.usdcSmallestUnit || this.usdcSmallestUnit === '0') {
                        throw new Error('Invalid USDC conversion. Please try again.');
                    }

                    elements.cryptoAmountEl.text(data.usdc_amount.toFixed(2) + ' ' + symbol);
                    elements.rateEl.text('1 ' + symbol + ' = 1.00 USD (Rate: 1 ' + data.fiat_currency + ' = ' + data.exchange_rate.toFixed(4) + ' USD)');
                } else {
                    this.ethAmount = data.eth_amount;
                    this.weiAmount = data.wei_amount;

                    // Validate ETH conversion result
                    if (!this.weiAmount || this.weiAmount === '0') {
                        throw new Error('Invalid ETH conversion. Please try again.');
                    }

                    elements.cryptoAmountEl.text(data.eth_amount.toFixed(6) + ' ' + symbol);
                    elements.rateEl.text('1 ' + symbol + ' = ' + data.eth_price.toFixed(2) + ' ' + data.fiat_currency);
                }

                elements.priceDisplay.show();
                elements.cryptoPaidInput.val(this.paymentType === 'usdc' ? this.usdcAmount : this.ethAmount);

                this.hideError();
                this.updateButtonText();

            } catch (error) {
                console.error('LCC: Conversion error:', error);
                this.showError(lccData.i18n.conversionError);
            }
        },

        /**
         * Validate the checkout form before payment
         */
        validateForm: function() {
            const $form = $('form.checkout');

            $form.find('.validate-required').each(function() {
                const $field = $(this);
                const $input = $field.find('input, select, textarea').not('[type="hidden"]');

                if ($input.length && !$input.val()) {
                    $field.addClass('woocommerce-invalid woocommerce-invalid-required-field');
                } else {
                    $field.removeClass('woocommerce-invalid woocommerce-invalid-required-field');
                }
            });

            const hasErrors = $form.find('.woocommerce-invalid').length > 0;

            if (hasErrors) {
                const $firstError = $form.find('.woocommerce-invalid').first();
                if ($firstError.length) {
                    $('html, body').animate({
                        scrollTop: $firstError.offset().top - 100
                    }, 500);
                }
                this.showError('Please fill in all required fields before paying.');
                return false;
            }

            const $email = $('#billing_email');
            if ($email.length && $email.val()) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailRegex.test($email.val())) {
                    this.showError('Please enter a valid email address.');
                    return false;
                }
            }

            return true;
        },

        /**
         * Process the payment
         */
        processPayment: async function() {
            const self = this;
            const elements = this.getElements();

            if (!this.validateForm()) {
                return;
            }

            if (!this.provider) {
                this.showError('Wallet not connected. Please reconnect.');
                return;
            }

            try {
                this.setButtonState('processing');
                this.hideError();

                // Refresh conversion with latest price
                await this.fetchConversion();

                // Check if we have valid amounts after conversion
                if (this.paymentType === 'usdc' && !this.usdcSmallestUnit) {
                    throw new Error('Failed to get USDC conversion. Please try again.');
                }
                if (this.paymentType !== 'usdc' && !this.weiAmount) {
                    throw new Error('Failed to get ETH conversion. Please try again.');
                }

                // Check minimum payment amounts (contract requirements)
                const MIN_ETH_WEI = '100000000000000'; // 0.0001 ETH
                const MIN_USDC = '100000'; // 0.1 USDC (6 decimals)

                if (this.paymentType === 'usdc' && BigInt(this.usdcSmallestUnit) < BigInt(MIN_USDC)) {
                    throw new Error('Order total is below minimum (0.1 USDC). Please add more items.');
                }
                if (this.paymentType !== 'usdc' && BigInt(this.weiAmount) < BigInt(MIN_ETH_WEI)) {
                    throw new Error('Order total is below minimum (0.0001 ETH ≈ €0.35). Please add more items.');
                }

                const paymentId = Date.now();

                let txHash;
                if (this.paymentType === 'usdc') {
                    txHash = await this.processUsdcPayment(paymentId);
                } else {
                    this.setButtonState('confirming');
                    txHash = await this.sendTransaction(paymentId);
                }


                // Store the values in hidden fields BEFORE any form submission
                elements.txHashInput.val(txHash);
                elements.cryptoPaidInput.val(this.paymentType === 'usdc' ? this.usdcAmount : this.ethAmount);

                // Also store network and payment type
                $('#lcc-payment-type').val(this.paymentType);

                // Store the selected network
                const network = this.getSelectedNetwork();
                if (network && network.key) {
                    $('#lcc-network').val(network.key);
                }

                this.setButtonState('complete');

                // Use a short delay to ensure UI updates, then submit
                setTimeout(function() {
                    // Verify values are set before submitting
                    if (!elements.txHashInput.val()) {
                        console.error('LCC: TX hash not set before submit!');
                        self.showError('Payment completed but form submission failed. TX: ' + txHash);
                        return;
                    }


                    // For WooCommerce AJAX checkout, we need to trigger the checkout submission
                    // Using the standard jQuery submit() which WooCommerce intercepts
                    const $form = $('form.checkout');

                    if ($form.length) {
                        // Remove any existing error messages
                        $('.woocommerce-error').remove();

                        // Trigger WooCommerce checkout
                        $form.submit();
                    } else {
                        console.error('LCC: Checkout form not found');
                        self.showError('Checkout form not found. Please refresh and try again.');
                    }
                }, 500);

            } catch (error) {
                console.error('LCC: Payment error:', error);
                this.showError(error.message || lccData.i18n.transactionFailed);
                this.setButtonState('ready');
            }
        },

        /**
         * Process USDC payment (approve + transfer)
         */
        processUsdcPayment: async function(orderId) {
            const provider = this.provider;
            const network = this.getSelectedNetwork();
            const merchantAddress = lccData.merchantAddress;

            if (!merchantAddress || !merchantAddress.match(/^0x[a-fA-F0-9]{40}$/)) {
                throw new Error(lccData.i18n.merchantNotConfigured);
            }

            if (!network.usdcAddress) {
                throw new Error('USDC not available on this network');
            }

            const usdcAddress = network.usdcAddress;
            const contractAddress = network.contract;
            const amount = this.usdcSmallestUnit;

            // Check USDC balance first
            this.setButtonState('processing');
            const balance = await this.getUsdcBalance(usdcAddress);

            if (BigInt(balance) < BigInt(amount)) {
                throw new Error(lccData.i18n.insufficientUsdc);
            }

            // Check current allowance
            const currentAllowance = await this.getUsdcAllowance(usdcAddress, contractAddress);

            // If allowance is insufficient, request approval
            if (BigInt(currentAllowance) < BigInt(amount)) {
                this.setButtonState('approving');

                const approveTxHash = await this.approveUsdc(usdcAddress, contractAddress, amount);

                // Wait for approval confirmation
                await this.waitForTransaction(approveTxHash);
            }

            // Now send the payment
            this.setButtonState('confirming');
            const txHash = await this.sendTokenTransaction(orderId, usdcAddress, amount);

            await this.waitForTransaction(txHash);

            return txHash;
        },

        /**
         * Get USDC balance
         */
        getUsdcBalance: async function(usdcAddress) {
            const provider = this.provider;

            // balanceOf(address) selector: 0x70a08231
            const data = '0x70a08231' + this.account.slice(2).toLowerCase().padStart(64, '0');

            const result = await provider.request({
                method: 'eth_call',
                params: [{
                    to: usdcAddress,
                    data: data
                }, 'latest']
            });

            return result;
        },

        /**
         * Get USDC allowance
         */
        getUsdcAllowance: async function(usdcAddress, spender) {
            const provider = this.provider;

            // allowance(address,address) selector: 0xdd62ed3e
            const data = '0xdd62ed3e' +
                this.account.slice(2).toLowerCase().padStart(64, '0') +
                spender.slice(2).toLowerCase().padStart(64, '0');

            const result = await provider.request({
                method: 'eth_call',
                params: [{
                    to: usdcAddress,
                    data: data
                }, 'latest']
            });

            return result;
        },

        /**
         * Approve USDC spending
         */
        approveUsdc: async function(usdcAddress, spender, amount) {
            const provider = this.provider;

            // approve(address,uint256) selector: 0x095ea7b3
            // SECURITY: Only approve exact amount needed (with 10% buffer for price changes)
            const approvalAmount = BigInt(amount) * 110n / 100n; // +10% buffer
            const data = '0x095ea7b3' +
                spender.slice(2).toLowerCase().padStart(64, '0') +
                approvalAmount.toString(16).padStart(64, '0');

            const txParams = {
                to: usdcAddress,
                from: this.account,
                data: data
            };

            // Estimate gas
            try {
                const gasEstimate = await provider.request({
                    method: 'eth_estimateGas',
                    params: [txParams]
                });
                txParams.gas = '0x' + Math.floor(parseInt(gasEstimate, 16) * 1.2).toString(16);
            } catch (e) {
                txParams.gas = '0x20000'; // 131072
            }

            const txHash = await provider.request({
                method: 'eth_sendTransaction',
                params: [txParams]
            });

            return txHash;
        },

        /**
         * Send token payment transaction
         */
        sendTokenTransaction: async function(orderId, tokenAddress, amount) {
            const provider = this.provider;
            const network = this.getSelectedNetwork();
            const merchantAddress = lccData.merchantAddress;

            // payWithToken(uint256,address,address,uint256) selector
            // keccak256("payWithToken(uint256,address,address,uint256)") = 0xd6bcaa76
            const functionSelector = '0xd6bcaa76';

            const encodedOrderId = orderId.toString(16).padStart(64, '0');
            const encodedMerchant = merchantAddress.slice(2).toLowerCase().padStart(64, '0');
            const encodedToken = tokenAddress.slice(2).toLowerCase().padStart(64, '0');
            const encodedAmount = BigInt(amount).toString(16).padStart(64, '0');

            const transactionParameters = {
                to: network.contract,
                from: this.account,
                data: functionSelector + encodedOrderId + encodedMerchant + encodedToken + encodedAmount,
            };

            // Estimate gas
            try {
                const gasEstimate = await provider.request({
                    method: 'eth_estimateGas',
                    params: [transactionParameters],
                });
                transactionParameters.gas = '0x' + Math.floor(parseInt(gasEstimate, 16) * 1.2).toString(16);
            } catch (gasError) {
                transactionParameters.gas = '0x50000'; // 327680
            }


            const txHash = await provider.request({
                method: 'eth_sendTransaction',
                params: [transactionParameters],
            });

            return txHash;
        },

        /**
         * Send ETH transaction to smart contract
         */
        sendTransaction: async function(orderId) {
            const provider = this.provider;
            const network = this.getSelectedNetwork();
            const merchantAddress = lccData.merchantAddress;

            if (!merchantAddress || !merchantAddress.match(/^0x[a-fA-F0-9]{40}$/)) {
                throw new Error(lccData.i18n.merchantNotConfigured);
            }

            // pay(uint256,address) function selector: 0x31cbf5e3
            const functionSelector = '0x31cbf5e3';
            const encodedOrderId = orderId.toString(16).padStart(64, '0');
            const encodedMerchant = merchantAddress.slice(2).toLowerCase().padStart(64, '0');

            const transactionParameters = {
                to: network.contract,
                from: this.account,
                value: '0x' + BigInt(this.weiAmount).toString(16),
                data: functionSelector + encodedOrderId + encodedMerchant,
            };

            try {
                const gasEstimate = await provider.request({
                    method: 'eth_estimateGas',
                    params: [transactionParameters],
                });
                const gasLimit = Math.floor(parseInt(gasEstimate, 16) * 1.2);
                transactionParameters.gas = '0x' + gasLimit.toString(16);
            } catch (gasError) {
                transactionParameters.gas = '0x30000';
            }


            const txHash = await provider.request({
                method: 'eth_sendTransaction',
                params: [transactionParameters],
            });

            await this.waitForTransaction(txHash);

            return txHash;
        },

        /**
         * Wait for transaction to be mined
         */
        waitForTransaction: async function(txHash, maxAttempts) {
            const provider = this.provider;
            maxAttempts = maxAttempts || 60;

            for (let i = 0; i < maxAttempts; i++) {
                const receipt = await provider.request({
                    method: 'eth_getTransactionReceipt',
                    params: [txHash],
                });

                if (receipt) {
                    if (receipt.status === '0x1') {
                        return receipt;
                    } else {
                        throw new Error('Transaction failed');
                    }
                }

                await new Promise(function(resolve) { setTimeout(resolve, 2000); });
            }

            throw new Error('Transaction confirmation timeout');
        },

        /**
         * Get cart total from page
         */
        getCartTotal: function() {
            // Try multiple selectors for compatibility with different themes
            const selectors = [
                '.order-total .woocommerce-Price-amount bdi',
                '.order-total .woocommerce-Price-amount',
                '.order-total td .amount',
                '.order-total .amount',
                'tr.order-total td:last-child',
                '.cart-subtotal .woocommerce-Price-amount bdi',
                '.cart-subtotal .woocommerce-Price-amount',
                '.cart_totals .order-total .amount',
                '#order_review .order-total .amount'
            ];

            let totalEl = null;
            for (let i = 0; i < selectors.length; i++) {
                totalEl = $(selectors[i]).first();
                if (totalEl.length && totalEl.text().trim()) {
                    break;
                }
                totalEl = null;
            }

            if (totalEl && totalEl.length) {
                let text = totalEl.text();
                // Remove currency symbols, spaces, and normalize decimal separator
                // Handle both comma and dot as decimal separators
                text = text.replace(/[^\d.,]/g, '');

                // If there's both comma and dot, determine which is decimal separator
                // European format: 1.234,56 -> American format: 1234.56
                if (text.indexOf(',') > text.indexOf('.')) {
                    // Comma is decimal separator (European format)
                    text = text.replace(/\./g, '').replace(',', '.');
                } else {
                    // Dot is decimal separator (American format)
                    text = text.replace(/,/g, '');
                }

                const value = parseFloat(text);
                return value || 0;
            }

            // Fallback: try to get from lccData if available
            if (typeof lccData !== 'undefined' && lccData.cartTotal) {
                return parseFloat(lccData.cartTotal) || 0;
            }

            return 0;
        },

        /**
         * Set button state
         */
        setButtonState: function(state) {
            const btn = this.getElements().connectBtn;
            const network = this.getSelectedNetwork();
            const symbol = this.paymentType === 'usdc' ? 'USDC' : (network ? network.symbol : 'ETH');

            switch(state) {
                case 'connect':
                    btn.prop('disabled', false)
                       .removeClass('processing complete')
                       .html('<span class="lcc-btn-icon">&#129418;</span> ' + lccData.i18n.connectWallet);
                    break;

                case 'connecting':
                    btn.prop('disabled', true)
                       .addClass('processing')
                       .html('<span class="lcc-spinner"></span> ' + (lccData.i18n.connecting || 'Connecting...'));
                    break;

                case 'ready':
                    btn.prop('disabled', false)
                       .removeClass('processing')
                       .html('<span class="lcc-btn-icon">&#128176;</span> Pay with ' + symbol);
                    break;

                case 'processing':
                    btn.prop('disabled', true)
                       .addClass('processing')
                       .html('<span class="lcc-spinner"></span> ' + lccData.i18n.processing);
                    break;

                case 'approving':
                    btn.prop('disabled', true)
                       .addClass('processing')
                       .html('<span class="lcc-spinner"></span> ' + lccData.i18n.approving);
                    break;

                case 'confirming':
                    btn.prop('disabled', true)
                       .addClass('processing')
                       .html('<span class="lcc-spinner"></span> ' + lccData.i18n.waitingConfirmation);
                    break;

                case 'complete':
                    btn.prop('disabled', true)
                       .removeClass('processing')
                       .addClass('complete')
                       .html('<span class="lcc-btn-icon">&#9989;</span> ' + lccData.i18n.paymentComplete);
                    break;
            }
        },

        /**
         * Show error message
         */
        showError: function(message) {
            this.getElements().errorEl.text(message).show();
        },

        /**
         * Hide error message
         */
        hideError: function() {
            this.getElements().errorEl.hide().text('');
        },

        /**
         * Disconnect wallet
         */
        disconnect: function() {
            this.isConnected = false;
            this.account = null;
            this.ethAmount = null;
            this.weiAmount = null;
            this.usdcAmount = null;
            this.usdcSmallestUnit = null;

            // Disconnect WalletConnect provider if applicable
            if (this.wcProvider && this.wcProvider.disconnect) {
                try {
                    this.wcProvider.disconnect();
                } catch (e) {
                    // Ignore disconnect errors
                }
            }

            this.connectionType = null;
            this.wcProvider = null;
            this.provider = null;

            this.updateUI();
            this.getElements().priceDisplay.hide();
            this.setButtonState('connect');
        },

        /**
         * Validate payment before form submission
         */
        validatePayment: function() {
            const txHash = this.getElements().txHashInput.val();

            if (!txHash) {
                this.showError('Please complete the payment first.');
                return false;
            }

            return true;
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        LCC.init();
    });

    // Expose globally for debugging
    window.LCC = LCC;

})(jQuery);
